<?php
/**
 * Plugin Name: Tahc Hybrid Chat
 * Plugin URI: https://tahc.ai/wordpress
 * Description: Add AI-powered hybrid chat to your WordPress site. Combines the speed of AI with the empathy of real humans.
 * Version: 1.0.1
 * Author: Tahc
 * Author URI: https://tahc.ai
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: tahc-chat
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('TAHC_CHAT_VERSION', '1.0.1');
define('TAHC_CHAT_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('TAHC_CHAT_PLUGIN_URL', plugin_dir_url(__FILE__));

/**
 * Main Tahc Chat Widget Class
 */
class Tahc_Chat_Widget {

    /**
     * Constructor
     */
    public function __construct() {
        // Add settings page to admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));

        // Register settings
        add_action('admin_init', array($this, 'register_settings'));

        // Add widget to frontend
        add_action('wp_footer', array($this, 'add_chat_widget'));

        // Add settings link on plugins page
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'add_settings_link'));
    }

    /**
     * Add admin menu page
     */
    public function add_admin_menu() {
        add_options_page(
            'Tahc Chat Settings',
            'Tahc Chat',
            'manage_options',
            'tahc-chat-settings',
            array($this, 'settings_page')
        );
    }

    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('tahc_chat_settings', 'tahc_workspace_id', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => ''
        ));

        register_setting('tahc_chat_settings', 'tahc_chat_enabled', array(
            'type' => 'boolean',
            'sanitize_callback' => 'rest_sanitize_boolean',
            'default' => true
        ));
    }

    /**
     * Settings page HTML
     */
    public function settings_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            return;
        }

        // Get workspace ID
        $workspace_id = get_option('tahc_workspace_id', '');
        $chat_enabled = get_option('tahc_chat_enabled', true);

        // Show success message if settings saved
        if (isset($_GET['settings-updated'])) {
            add_settings_error('tahc_chat_messages', 'tahc_chat_message', 'Settings Saved', 'updated');
        }

        settings_errors('tahc_chat_messages');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>

            <div style="background: linear-gradient(135deg, #000000 0%, #333333 100%); color: white; padding: 20px; border-radius: 8px; margin: 20px 0;">
                <h2 style="color: white; margin-top: 0;">Welcome to Tahc Hybrid Chat! 🚀</h2>
                <p style="font-size: 16px; margin-bottom: 10px;">
                    Combine the speed of AI with the empathy of real humans to provide exceptional customer support.
                </p>
                <p style="font-size: 14px; margin-bottom: 0; opacity: 0.9;">
                    <strong>Free to try!</strong> Start with our 14-day free trial - no credit card required.
                </p>
            </div>

            <div style="background: white; padding: 20px; border: 1px solid #ccc; border-radius: 8px; margin-bottom: 20px;">
                <h2 style="margin-top: 0;">Quick Setup</h2>

                <?php if (empty($workspace_id)): ?>
                    <div style="background: #fff3cd; border-left: 4px solid #ffc107; padding: 15px; margin-bottom: 20px;">
                        <strong>📝 First Time Setup:</strong>
                        <ol style="margin: 10px 0 0 20px;">
                            <li>Don't have an account? <a href="https://dash.tahc.ai/signup" target="_blank" style="color: #667eea; font-weight: bold;">Sign up for free at tahc.ai</a></li>
                            <li>After signup, copy your Workspace ID from your dashboard</li>
                            <li>Paste it below and click "Save Changes"</li>
                        </ol>
                    </div>
                <?php else: ?>
                    <div style="background: #d4edda; border-left: 4px solid #28a745; padding: 15px; margin-bottom: 20px;">
                        <strong>✅ Chat Widget Active!</strong>
                        <p style="margin: 10px 0 0 0;">Your Tahc chat widget is installed and ready. Visit your site to see it in action!</p>
                    </div>
                <?php endif; ?>

                <form method="post" action="options.php">
                    <?php
                    settings_fields('tahc_chat_settings');
                    do_settings_sections('tahc_chat_settings');
                    ?>

                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="tahc_workspace_id">Workspace ID</label>
                            </th>
                            <td>
                                <input
                                    type="text"
                                    id="tahc_workspace_id"
                                    name="tahc_workspace_id"
                                    value="<?php echo esc_attr($workspace_id); ?>"
                                    class="regular-text"
                                    placeholder="your-company"
                                />
                                <p class="description">
                                    Your unique workspace identifier from your Tahc dashboard.
                                    <a href="https://dash.tahc.ai/dashboard" target="_blank">Find it here →</a>
                                </p>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row">
                                <label for="tahc_chat_enabled">Enable Chat Widget</label>
                            </th>
                            <td>
                                <label>
                                    <input
                                        type="checkbox"
                                        id="tahc_chat_enabled"
                                        name="tahc_chat_enabled"
                                        value="1"
                                        <?php checked($chat_enabled, true); ?>
                                    />
                                    Show chat widget on your website
                                </label>
                            </td>
                        </tr>
                    </table>

                    <?php submit_button('Save Changes'); ?>
                </form>
            </div>

            <div style="background: white; padding: 20px; border: 1px solid #ccc; border-radius: 8px; margin-bottom: 20px;">
                <h2 style="margin-top: 0;">What is Hybrid Chat?</h2>
                <p style="font-size: 15px; line-height: 1.6;">
                    <strong>Hybrid Chat</strong> blends the speed of AI with the empathy of real humans.
                    Your customers get instant answers from AI, with seamless handoff to human agents when needed.
                </p>

                <h3>Features:</h3>
                <ul style="line-height: 1.8; font-size: 15px;">
                    <li>🤖 <strong>AI-Powered Responses</strong> - Instant answers trained on your content</li>
                    <li>👤 <strong>Human Handoff</strong> - Seamlessly escalate to real people</li>
                    <li>🎭 <strong>Orchestrations</strong> - Custom AI behavior based on page, visitor, or intent</li>
                    <li>🌍 <strong>Multi-Language</strong> - Auto-detects and responds in 10+ languages</li>
                    <li>🎨 <strong>Fully Customizable</strong> - Match your brand perfectly</li>
                    <li>📱 <strong>Mobile Responsive</strong> - Works beautifully on all devices</li>
                    <li>🕐 <strong>Office Hours</strong> - Set your availability schedule</li>
                    <li>📊 <strong>Analytics</strong> - Track conversations and performance</li>
                    <li>📲 <strong>iOS App</strong> - Respond to customers from anywhere</li>
                </ul>
            </div>

            <div style="background: white; padding: 20px; border: 1px solid #ccc; border-radius: 8px; margin-bottom: 20px;">
                <h2 style="margin-top: 0;">Pricing</h2>
                <p style="font-size: 15px; margin-bottom: 20px;">
                    <strong>14-day free trial</strong> on all plans. No credit card required.
                </p>
                <div style="display: flex; gap: 15px; flex-wrap: wrap;">
                    <div style="flex: 1; min-width: 180px; border: 2px solid #e5e7eb; border-radius: 8px; padding: 15px; text-align: center;">
                        <h4 style="margin: 0 0 8px 0; color: #6b7280;">Starter</h4>
                        <p style="font-size: 24px; font-weight: bold; margin: 0 0 5px 0;">$19<span style="font-size: 14px; font-weight: normal;">/mo</span></p>
                        <p style="font-size: 12px; color: #6b7280; margin: 0;">100 articles</p>
                    </div>
                    <div style="flex: 1; min-width: 180px; border: 2px solid #000; border-radius: 8px; padding: 15px; text-align: center; background: #f9fafb;">
                        <h4 style="margin: 0 0 8px 0; color: #000;">Starter+ <span style="background: #000; color: #fff; font-size: 9px; padding: 2px 5px; border-radius: 4px;">POPULAR</span></h4>
                        <p style="font-size: 24px; font-weight: bold; margin: 0 0 5px 0;">$39<span style="font-size: 14px; font-weight: normal;">/mo</span></p>
                        <p style="font-size: 12px; color: #6b7280; margin: 0;">500 articles, 5 orchestrations</p>
                    </div>
                    <div style="flex: 1; min-width: 180px; border: 2px solid #e5e7eb; border-radius: 8px; padding: 15px; text-align: center;">
                        <h4 style="margin: 0 0 8px 0; color: #6b7280;">Pro</h4>
                        <p style="font-size: 24px; font-weight: bold; margin: 0 0 5px 0;">$79<span style="font-size: 14px; font-weight: normal;">/mo</span></p>
                        <p style="font-size: 12px; color: #6b7280; margin: 0;">White-label, 4 seats included</p>
                    </div>
                </div>
                <p style="font-size: 13px; color: #6b7280; margin-top: 15px; text-align: center;">
                    All plans include unlimited conversations, AI responses, human handoff, and analytics.
                </p>
            </div>

            <div style="background: #f8f9fa; padding: 20px; border-radius: 8px; text-align: center;">
                <h3 style="margin-top: 0;">Need Help?</h3>
                <p style="font-size: 15px;">
                    Visit our <a href="https://tahc.ai/docs" target="_blank" style="color: #667eea; font-weight: bold;">documentation</a>
                    or <a href="https://dash.tahc.ai/support" target="_blank" style="color: #667eea; font-weight: bold;">contact support</a>
                </p>
                <p style="margin: 20px 0 0 0;">
                    <a href="https://dash.tahc.ai/dashboard" target="_blank" class="button button-primary button-hero">
                        Open Tahc Dashboard →
                    </a>
                </p>
            </div>
        </div>
        <?php
    }

    /**
     * Add chat widget to frontend
     */
    public function add_chat_widget() {
        // Check if enabled
        $chat_enabled = get_option('tahc_chat_enabled', true);
        if (!$chat_enabled) {
            return;
        }

        // Get workspace ID
        $workspace_id = get_option('tahc_workspace_id', '');

        // Don't add if no workspace ID
        if (empty($workspace_id)) {
            return;
        }

        // Output the widget script
        ?>
        <!-- Tahc Hybrid Chat Widget -->
        <script src="https://app.tahc.ai/widget.js" data-workspace-id="<?php echo esc_attr($workspace_id); ?>"></script>
        <?php
    }

    /**
     * Add settings link to plugins page
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="options-general.php?page=tahc-chat-settings">Settings</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
}

// Initialize the plugin
new Tahc_Chat_Widget();
